"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.phoneToJid = void 0;
const Error_1 = require("../Error");
const countryCodes_1 = require("./countryCodes");
/**
 * Validates if a phone number starts with a valid country code.
 * @param phone - The phone number to validate.
 * @returns True if the phone number is valid, otherwise false.
 */
const isPhoneNumberValidCountry = (phone) => {
    if (!phone) {
        throw new Error_1.WhatsappError('parameter "phone" is required');
    }
    return countryCodes_1.COUNTRY_CODES.some((code) => phone.startsWith(code));
};
/**
 * Converts a phone number to a JID (Jabber ID) for WhatsApp messaging.
 * @param params - Parameters containing phone number and group flag.
 * @returns A valid JID string.
 */
const phoneToJid = ({ to, isGroup = false, }) => {
    if (!to) {
        throw new Error_1.WhatsappError('parameter "to" is required');
    }
    let number = to.toString().trim();
    // Validate phone number format
    if (!isPhoneNumberValidCountry(number)) {
        throw new Error_1.WhatsappError("Phone number must start with a valid country code");
    }
    // Normalize number and append domain
    number = number.replace(/\s|[+]|[-]/gim, ""); // Remove spaces, '+' and '-' characters
    if (isGroup) {
        if (!number.endsWith("@g.us")) {
            number = `${number}@g.us`;
        }
    }
    else {
        if (!number.endsWith("@s.whatsapp.net")) {
            number = `${number}@s.whatsapp.net`;
        }
    }
    return number;
};
exports.phoneToJid = phoneToJid;
